/* FILE: mtrigger.c                             (D. Tottingham  12/26/90)

This is a collection of routines that manage the trigger for xdetect.  The
low-level routines contained in this module (namely, triggered_on_channel
and energy_valid_on_channel) are based on the algorithm presented in
Lee's "Location and Real-Time Detection of Microearthquakes Along the San Andreas
Fault System in Central California."  The program has been written and compiled
medium model.  The following public functions are defined in this file:

t_display_event_params ()       write event parameters to stream
t_display_triggers ()           write event information to stream
t_display_trigger_params ()     write trigger parameters to stream
t_event_detected ()             check to see whether event is done
t_event_done ()                 return event status
t_get_channel_trigger_state ()  get trigger state for given channel
t_get_eventsetting ()           get T_EVENT structure
t_get_head_trigger ()           get trigger information at head of channel queue
t_get_last_trigger ()           get last trigger
t_get_next_trigger ()           get next trigger from channel queue
t_get_trigger_status ()         get trigger_enabled
t_get_trigger_time ()           get trigger time for a channel
t_get_trigsetting ()            get T_TRIGGER structure
t_initialize_event ()           initialize the event structure
t_initialize_trigger ()         initialize the trigger structure
t_reset_event ()                reset event variables
t_reset_trigger ()              reset trigger variables
t_set_CriticalAlpha ()          set CriticalAlpha constant
t_set_CriticalBeta ()           set CriticalBeta constant
t_set_CriticalGamma ()          set CriticalGamma constant
t_set_CriticalMu ()             set CriticalMu constant
t_set_CriticalNu ()             set CriticalNu constant
t_set_event ()                  set event detected flag to TRUE
t_set_EventContinueCount ()     set EventContinuationCount constant
t_set_LTAWindow ()              set length of long-term average
t_set_MaxEventTime ()           set MaxEventTime constant
t_set_MinEventTime ()           set MinEventTime constant
t_set_netwname ()               set network name
t_set_STAWindow ()              set length of short-term average
t_set_Trigger ()                set trigger status for given channel
t_set_TriggerConfirmCount ()    set TriggerConfirmationCount constant
t_set_TriggerTimeLimit ()       set TriggerTimeLimit constant
t_set_trigger_status ()         set trigger_enabled flag
t_toggle_trigger_status ()      toggle trigger_enabled flag
t_trigger_check ()              look for new triggers

EXTERNAL FUNCTIONS CALLED:

dm_get_first_buffer ()          get first new demux buffer from demux queue
dm_get_next_buffer ()           get next buffer from demux queue
h_set_trigger_status ()         set trigger status
q_delete_link ()                delete a data link from a data queue
q_enqueue ()                    enqueue a data link on a data queue
q_initialize ()                 initialize a data queue
q_insert_link ()                insert a data link from a data queue
s_display_triggers ()           display triggers on the screen
suds_initialize ()              initialize a suds structure
suds_initialize_tag ()          initialize a suds structtag
u_build_timeline ()             convert abs. time to an ascii string
u_ispow2 ()                     determine whether i_num is a power of two
u_log2 ()                       compute the base-2 logarithm of i_num
u_strncpy ()                    copies n characters of far string2 to far string1
u_timestamp ()                  get timestamp


HISTORY:
   2/21/91: removed first differnece from triggering (JR)
   2/11/92: changed from dx/new_lta to new_sta/new_lta


/*************************************************************************
                             INCLUDE FILES


*************************************************************************/
#include <ctype.h>
#include <io.h>
#include <malloc.h>
#include <math.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys\types.h>
#include <sys\timeb.h>
#include <time.h>

#include "mconst.h"
#include "mdemux.h"
#include "mdt28xx.h"
#include "merror.h"
#include "mqueue.h"
#include "mscreen.h"
#include "mscrnhdr.h"
#include "mstation.h"
#include "msudsini.h"
#include "mtrigger.h"
#include "mutils.h"

#define         ADC_MID_POINT                   2048

/*************************************************************************
                                GLOBALS


*************************************************************************/
PRIVATE T_EVENT eventsetting, far * e = &eventsetting;
PRIVATE T_TRIGGER trigsetting, far * t = &trigsetting;
PRIVATE Q_QUEUE channel_queue, trigger_queue;
PRIVATE Q_LINK *head_ptr;
PRIVATE Q_TRIGGER *lasttrig;
PRIVATE unsigned int trigger_location;
PRIVATE double last_trigger;
PRIVATE FLAG event_flag;
PRIVATE FLAG trigger_enabled;


/*=======================================================================*
 *                        energy_valid_on_channel                        *
 *=======================================================================*/
/* Check for energy on given channel.                                    */

PRIVATE
FLAG energy_valid_on_channel (tch, channel_buffer, n_pts)
Q_CHANNEL far * tch;
int far * channel_buffer;
unsigned n_pts;
{
   FLAG energy_valid;
   int dx, new_sta, new_lta;
   int gamma;
   unsigned k, inspect;

   /* Initialize sta/lta and flags */
   new_sta = (*tch).info.abs_sta;
   new_lta = (*tch).info.abs_lta;
   energy_valid = FALSE;
   inspect = 0;

   /* Main loop */
   for (k = 1; k < n_pts; k++) {
      /* Compute new sta */
      /* dx = abs (channel_buffer[k] - channel_buffer[k-1]); */
      dx = abs (channel_buffer[k] - ADC_MID_POINT);
      new_sta += (dx - new_sta) >> t->sta_shift;

      /* Compute new lta */
      new_lta += ((new_sta - new_lta) >> t->lta_shift);
      if (new_lta < LTA_INIT) new_lta = LTA_INIT;

      /* Do we still have an event? */
      if (energy_valid == FALSE) {
         if ((k - inspect) <= e->event_continuation_count) {
            gamma = new_sta / (*tch).info.lta;

            /* Do we have enough energy? */
            if (gamma < e->critical_gamma)
               inspect = k;
         }
         else energy_valid = TRUE;
      }
   }

   /* Store channel's sta/lta */
   (*tch).info.abs_sta = new_sta;
   (*tch).info.abs_lta = new_lta;

   return (energy_valid);
}

/*=======================================================================*
 *                           triggered_on_channel                        *
 *=======================================================================*/
/* Look for a trigger on the given channel.                              */

PRIVATE
FLAG triggered_on_channel (qch, channel_buffer, n_pts )
Q_CHANNEL far * qch;
int far * channel_buffer;
unsigned n_pts;
{
   int dx, ns, new_sta, new_lta;
   int alpha, beta;
   unsigned k;
   FLAG trigger_flag;

   /* Initialize sta and lta for this channel */
   new_sta = (*qch).info.abs_sta;
   new_lta = (*qch).info.abs_lta;

   /* Initialize trigger_time and flags */
   trigger_location = 0;
   trigger_flag = FALSE;

   /* Main loop */
   for (k = 1; k < n_pts; k++) {
      /* Compute new sta */
   /*   dx = abs (channel_buffer[k] - channel_buffer[k-1]);  */
      dx = abs (channel_buffer[k] - ADC_MID_POINT);
      new_sta += (dx - new_sta) >> t->sta_shift;

      /* Compute new lta */
      new_lta += (new_sta - new_lta) >> t->lta_shift;
      if (new_lta < LTA_INIT) new_lta = LTA_INIT;

      /* If we have not hit a trigger, continue checking */
      if (trigger_flag == FALSE) {
         /* Compute signal and noise */
         ns = abs (channel_buffer[k] - ADC_MID_POINT);
         (*qch).noise += (ns - (*qch).noise) >> 8;
         if ((*qch).noise < NOISE_INIT) (*qch).noise = NOISE_INIT;

	 /* Compute alpha */
/* changed from dx to new_sta */
	 alpha = new_sta / new_lta;

         /* Do we have a trigger? */
	 if ( alpha > t->critical_alpha) {
            trigger_flag = TRUE;
            trigger_location = k;
            (*qch).info.lta = new_lta;
            (*qch).info.sta = new_sta;
         } else continue;
      }

      /* We have hit a trigger.  Are we still within time window? */
      if ((k - trigger_location) <= t->trigger_confirmation_count) {

         /* We are within the time window so calculate beta */
         beta = new_sta / (*qch).info.lta;

         /* Do we have enough energy? */
         if (beta < t->critical_beta)
            trigger_flag = FALSE;
      }
   }

   /* Store channel's LTA and STA */
   (*qch).info.abs_sta = new_sta;
   (*qch).info.abs_lta = new_lta;

   return (trigger_flag);
}

/*=======================================================================*
 *                           trigger_window_check                        *
 *=======================================================================*/
/* Make sure that current triggers are still within the allowed trigger
   window.                                                               */

PRIVATE
void trigger_window_check (time)
double time;
{
   Q_LINK *current, *previous, *temp;
   Q_TYPE type;

   current = trigger_queue.head;
   previous = NULL;
   while (current != NULL) {
      if (fabs(time - current->type.trigger->trigger_time) >
               t->trigger_time_limit) {
         temp = current;
         current->type.trigger->qch->info.trig_value = OFF;
         current = current->next;
         if (q_delete_link (&trigger_queue, temp, previous, &type))
            free (type.trigger);
      } else {
         previous = current;
         current = current->next;
      }
   }
}

/*=======================================================================*
 *                        t_display_event_params                         *
 *=======================================================================*/
/* Write event parameters to the specified stream.                       */

PUBLIC
void t_display_event_params (stream)
FILE * stream;
{
   fprintf (stream, "min_event_time             = %5f\n", e->min_event_time);
   fprintf (stream, "max_event_time             = %5f\n", e->max_event_time);
   fprintf (stream, "event_continuation_count   = %5d\n", e->event_continuation_count);
   fprintf (stream, "critical_gamma             = %5d\n", e->critical_gamma);
   fprintf (stream, "critical_mu                = %5d\n", e->critical_mu);
}

/*=======================================================================*
 *                           t_display_triggers                          *
 *=======================================================================*/
/* Write event information to the specified stream.                      */

PUBLIC
void t_display_triggers (stream)
FILE * stream;
{
   Q_LINK * head;
   char * timeline;

   timeline = u_build_timeline (last_trigger, TRUE);

   fprintf (stream, "\n\n\nEVENT DETECTED      %s\n\n",
           timeline );
   fprintf (stream, "CH    AVG                      \n");
   fprintf (stream, "      Short   Long   Time      \n");
   fprintf (stream, "---   -----   -----  --------------------------- \n");

   free (timeline);

   head = channel_queue.head;
   while (head != NULL) {
      if (head->type.channel->info.trig_value) {
         timeline = u_build_timeline (head->type.channel->info.trig_time, TRUE);
         fprintf (stream, "%3d %c %4d    %4d  %s\n",
                  head->type.channel->channel,
                  (head->type.channel->info.trig_value == ON) ? ' ': 'E',
                  head->type.channel->info.sta,
                  head->type.channel->info.lta, timeline);
         free (timeline);
      }
      head = head->next;
   }
   fprintf (stream, "\n");
}

/*=======================================================================*
 *                        t_display_trigger_params                       *
 *=======================================================================*/
/* Write trigger parameters to specified stream.                         */

PUBLIC
void t_display_trigger_params (stream)
FILE * stream;
{
   fprintf (stream, "critical_alpha             = %5d\n", t->critical_alpha);
   fprintf (stream, "trigger_confirmation_count = %5d\n", t->trigger_confirmation_count);
   fprintf (stream, "critical_beta              = %5d\n", t->critical_beta);
   fprintf (stream, "trigger_time_limit         = %5f\n", t->trigger_time_limit);
   fprintf (stream, "critical_nu                = %5d\n", t->critical_nu);
   fprintf (stream, "lta_window                 = %5d\n", t->lta_window);
   fprintf (stream, "sta_window                 = %5d\n", t->sta_window);
}

/*=======================================================================*
 *                               t_event_done                            *
 *=======================================================================*/
/* Check to see whether event is done.  A minimum post trigger window
   is saved regardless of energy content.  Past this point until the
   maximum post trigger window is reached, the event must pass the
   energy_valid_on_channel test.                                         */

PUBLIC
FLAG t_event_done ()
{
   Q_BUFFER far * b_head;
   Q_LINK * c_head;
   int far * channel_buffer;
   unsigned channel_ctr, i;

   b_head = dm_get_first_buffer();
   while (b_head != NULL) {
      if ((b_head->info.begintime - last_trigger) < e->min_event_time) {
         /* Just update the sta and lta */
         c_head = channel_queue.head;
         while (c_head != NULL) {
            channel_buffer = b_head->data + b_head->info.blocksize * c_head->type.channel->channel;
            energy_valid_on_channel (c_head->type.channel, channel_buffer, b_head->info.blocksize);
            c_head = c_head->next;
         }
      } else if ((b_head->info.begintime - last_trigger) < e->max_event_time) {
         /* Scan buffer for any activity */
         channel_ctr = 0;
         c_head = channel_queue.head;
         while (c_head != NULL) {
            channel_buffer = b_head->data + b_head->info.blocksize * c_head->type.channel->channel;
            if (energy_valid_on_channel (c_head->type.channel, channel_buffer, b_head->info.blocksize))
               channel_ctr++;
            c_head = c_head->next;
         }
         event_flag = (channel_ctr < e->critical_mu) ? FALSE : TRUE;
         if (!event_flag) return ( TRUE);
      } else {
         event_flag = FALSE;
         return ( TRUE);
      }
      b_head = dm_get_next_buffer ();
   }
   return ( FALSE);
}

/*=======================================================================*
 *                           t_event_detected                            *
 *=======================================================================*/
/* Return event status.                                                  */

PUBLIC
FLAG t_event_detected ()
{
   return (event_flag);
}

/*=======================================================================*
 *                     t_get_channel_trigger_state                       *
 *=======================================================================*/
/* Get trigger state for given channel.                                  */

PUBLIC
unsigned int t_get_channel_trigger_state (channel)
unsigned int channel;
{
   Q_LINK * head;

   head = channel_queue.head;
   while (head != NULL) {
      if (head->type.channel->channel == channel)
         return (head->type.channel->info.trig_value + 1);
      head = head->next;
   }
   return CHANNEL_TRIGGER_DISABLED;
}

/*=======================================================================*
 *                          t_get_eventsetting                           *
 *=======================================================================*/
/* Get T_EVENT structure.                                                */

PUBLIC
T_EVENT far * t_get_eventsetting ()
{
   return (e);
}

/*=======================================================================*
 *                         t_get_head_trigger                            *
 *=======================================================================*/
/* Get trigger information at head of channel queue.                     */

PUBLIC
Q_CHANNEL far * t_get_head_trigger ()
{
   head_ptr = channel_queue.head;
   if (head_ptr != NULL)
      return (head_ptr->type.channel);
   else return (NULL);
}

/*=======================================================================*
 *                          t_get_last_trigger                           *
 *=======================================================================*/
/* Get last trigger.                                                     */

PUBLIC
double t_get_last_trigger ()
{
   return (last_trigger);
}

/*=======================================================================*
 *                        t_get_next_trigger                             *
 *=======================================================================*/
/* Get next trigger from channel queue.                                  */

PUBLIC
Q_CHANNEL far * t_get_next_trigger ()
{
   head_ptr = head_ptr->next;
   if (head_ptr != NULL)
      return (head_ptr->type.channel);
   else return (NULL);
}

/*=======================================================================*
 *                         t_get_trigger_status                          *
 *=======================================================================*/
/* Get trigger_enabled.                                                  */

PUBLIC
FLAG t_get_trigger_status ()
{
   return trigger_enabled;
}

/*=======================================================================*
 *                          t_get_trigger_time                           *
 *=======================================================================*/
/* Get trigger time for specified channel.                               */

PUBLIC
double t_get_trigger_time (channel)
unsigned int channel;
{
   Q_LINK * head;

   head = channel_queue.head;
   while (head != NULL) {
      if (head->type.channel->channel == channel)
         return (head->type.channel->info.trig_time);
      head = head->next;
   }
   er_abort (T_INVALID_CHANNEL);
}

/*=======================================================================*
 *                          t_get_trigsetting                            *
 *=======================================================================*/
/* Return T_TRIGGER structure.                                           */

PUBLIC
T_TRIGGER far * t_get_trigsetting ()
{
   return (t);
}

/*=======================================================================*
 *                          t_initialize_event                           *
 *=======================================================================*/
/* Initialize the EVENT_INFO structure.                                  */

PUBLIC
FLAG t_initialize_event ()
{
   suds_initialize_tag (EVENTSETTING, &(e->structtag));
   suds_initialize (EVENTSETTING, &(e->eventsetting));

   e->eventsetting.algorithm = 'x';
   u_strncpy (e->eventsetting.netwname, (char far *)(st_get_netwname()), 4);

   e->critical_gamma = CRITICAL_GAMMA;
   e->critical_mu = CRITICAL_MU;
   e->event_continuation_count = EVENT_CONTINUATION_COUNT;
   e->max_event_time = MAX_EVENT_TIME;
   e->min_event_time = MIN_EVENT_TIME;
   e->eventsetting.beginttime = u_timestamp ();

   event_flag = FALSE;
}

/*=======================================================================*
 *                          t_initialize_trigger                         *
 *=======================================================================*/
/* Initialize the TRIGGER_INFO structure.                                */

PUBLIC
FLAG t_initialize_trigger ()
{
   trigger_enabled = TRIGGER_ENABLED;
   h_set_trigger_status (trigger_enabled);

   suds_initialize_tag (TRIGSETTING, &(t->structtag));
   suds_initialize (TRIGSETTING, &(t->trigsetting));

   t->trigsetting.algorithm = 'x';
   u_strncpy (t->trigsetting.netwname, (char far *)(st_get_netwname()), 4);

   t->critical_alpha = CRITICAL_ALPHA;
   t->critical_beta = CRITICAL_BETA;
   t->critical_nu = CRITICAL_NU;
   t->trigger_confirmation_count = TRIGGER_CONFIRMATION_COUNT;
   t->trigger_time_limit = TRIGGER_TIME_LIMIT;
   t->lta_shift = LTA_SHIFT;
   t->sta_shift = STA_SHIFT;
   t->lta_window = (unsigned int) pow (((double) t->lta_shift), 2.0);
   t->sta_window = (unsigned int) pow (((double) t->sta_shift), 2.0);
   t->trigsetting.beginttime = u_timestamp ();

   q_initialize (&channel_queue);
   q_initialize (&trigger_queue);
   last_trigger = 0;
   lasttrig = NULL;

   return( TRUE);
}

/*=======================================================================*
 *                            t_reset_event                              *
 *=======================================================================*/
/* Reset event variables.                                                */

PUBLIC
void t_reset_event ()
{
   event_flag = FALSE;
}

/*=======================================================================*
 *                            t_reset_trigger                            *
 *=======================================================================*/
/* Reset trigger variables.                                              */

PUBLIC
void t_reset_trigger ()
{
   Q_LINK * head;

   head = channel_queue.head;
   while (head != NULL) {
      head->type.channel->info.abs_lta = head->type.channel->info.abs_sta;
      head = head->next;
   }
}

/*=======================================================================*
 *                           t_set_CriticalAlpha                         *
 *=======================================================================*/
/* Set CriticalAlpha constant.                                           */

PUBLIC
void t_set_CriticalAlpha (c)
int c;
{
   t->critical_alpha = c;
   t->trigsetting.beginttime = u_timestamp ();
}

/*=======================================================================*
 *                           t_set_CriticalBeta                          *
 *=======================================================================*/
/* Set CriticalBeta constant.                                            */

PUBLIC
void t_set_CriticalBeta (c)
int c;
{
   t->critical_beta = c;
   t->trigsetting.beginttime = u_timestamp ();
}

/*=======================================================================*
 *                          t_set_CriticalGamma                          *
 *=======================================================================*/
/* Set CriticalGamma constant.                                           */

PUBLIC
void t_set_CriticalGamma (c)
int c;
{
   e->critical_gamma = c;
   e->eventsetting.beginttime = u_timestamp ();
}

/*=======================================================================*
 *                            t_set_CriticalMu                           *
 *=======================================================================*/
/* Set CriticalMu constant.                                              */

PUBLIC
void t_set_CriticalMu (c)
int c;
{
   e->critical_mu = c;
   e->eventsetting.beginttime = u_timestamp ();
}

/*=======================================================================*
 *                            t_set_CriticalNu                           *
 *=======================================================================*/
/* Set CriticalNu constant.                                              */

PUBLIC
void t_set_CriticalNu (c)
int c;
{
   t->critical_nu = c;
   t->trigsetting.beginttime = u_timestamp ();
}

/*=======================================================================*
 *                              t_set_event                              *
 *=======================================================================*/
/* Set event detected flag to TRUE.                                      */

PUBLIC
void t_set_event ()
{
   event_flag = TRUE;
}

/*=======================================================================*
 *                         t_set_EventContinueCount                      *
 *=======================================================================*/
/* Set EventContinuationCount constant.                                  */

PUBLIC
void t_set_EventContinueCount (count)
int count;
{
   e->event_continuation_count = count;
   e->eventsetting.beginttime = u_timestamp ();
}

/*=======================================================================*
 *                              t_set_LTAWindow                          *
 *=======================================================================*/
/* Set length of long-term average.                                      */

PUBLIC
void t_set_LTAWindow (length)
unsigned int length;
{
   if (! u_ispow2 ((unsigned long) length))
      er_abort (T_INVALID_LTASIZE);
   t->lta_window = length;
   t->lta_shift = (unsigned int) u_log2 ((unsigned long) length);
}

/*=======================================================================*
 *                            t_set_MaxEventTime                         *
 *=======================================================================*/
/* Set MaxEventTime constant.                                            */

PUBLIC
void t_set_MaxEventTime (m)
double m;
{
   e->max_event_time = m;
   e->eventsetting.beginttime = u_timestamp ();
}

/*=======================================================================*
 *                            t_set_MinEventTime                         *
 *=======================================================================*/
/* Set MinEventTime constant.                                            */

PUBLIC
void t_set_MinEventTime (m)
double m;
{
   e->min_event_time = m;
   e->eventsetting.beginttime = u_timestamp ();
}

/*=======================================================================*
 *                             t_set_netwname                            *
 *=======================================================================*/
/* Set network name.                                                     */

PUBLIC
void t_set_netwname (nwn)
char nwn[];
{
   u_strncpy (t->trigsetting.netwname, (char far *)nwn, 4);
   u_strncpy (e->eventsetting.netwname, (char far *)nwn, 4);
   t->trigsetting.beginttime = u_timestamp ();
   e->eventsetting.beginttime = u_timestamp ();
}

/*=======================================================================*
 *                              t_set_STAWindow                          *
 *=======================================================================*/
/* Set length of short-term average.                                     */

PUBLIC
void t_set_STAWindow (length)
unsigned int length;
{
   if (! u_ispow2 ((unsigned long) length))
      er_abort (T_INVALID_STASIZE);
   t->sta_window = length;
   t->sta_shift = (unsigned int) u_log2 ((unsigned long) length);
}

/*=======================================================================*
 *                              t_set_Trigger                            *
 *=======================================================================*/
/* Set trigger status for given channel.                                 */

PUBLIC
void t_set_Trigger (channel, trigger_status)
unsigned int channel;
FLAG trigger_status;
{
   Q_CHANNEL far * qch;
   Q_TYPE type;

   if (trigger_status) {
      qch = (Q_CHANNEL far *) _fmalloc (sizeof(Q_CHANNEL));
      if (qch == NULL) er_abort (T_NO_STORAGE);

      suds_initialize_tag (TRIGGERS, &qch->structtag);
      suds_initialize (TRIGGERS, &qch->info);

      qch->channel = channel;
      qch->signal = 0;
      qch->noise = NOISE_INIT;
      qch->info.tr_name = st_get_station(channel)->info.sc_name;
      qch->info.trig_value = FALSE;
      qch->info.abs_sta = 0;
      qch->info.abs_lta = LTA_INIT;
      type.channel = qch;
      q_enqueue (&channel_queue, type);
   }
}

/*=======================================================================*
 *                        t_set_TriggerConfirmCount                      *
 *=======================================================================*/
/* Set TriggerConfirmationCount constant.                                */

PUBLIC
void t_set_TriggerConfirmCount (count)
int count;
{
   t->trigger_confirmation_count = count;
   t->trigsetting.beginttime = u_timestamp ();
}

/*=======================================================================*
 *                          t_set_TriggerTimeLimit                       *
 *=======================================================================*/
/* Set TriggerTimeLimit constant.                                        */

PUBLIC
void t_set_TriggerTimeLimit (timelimit)
double timelimit;
{
   t->trigger_time_limit = timelimit;
   t->trigsetting.beginttime = u_timestamp ();
}

/*=======================================================================*
 *                          t_set_trigger_status                         *
 *=======================================================================*/
/* Set trigger_enabled flag.                                             */

PUBLIC
void t_set_trigger_status (trigger_status)
FLAG trigger_status;
{
   trigger_enabled = trigger_status;
}

/*=======================================================================*
 *                        t_toggle_trigger_status                        *
 *=======================================================================*/
/* Toggle the trigger_enabled flag.                                      */

PUBLIC
void t_toggle_trigger_status ()
{
   trigger_enabled = (trigger_enabled) ? FALSE : TRUE;
}

/*=======================================================================*
 *                             t_trigger_check                           *
 *=======================================================================*/
/* Look for new triggers on all participating channels.                  */

PUBLIC
FLAG t_trigger_check ()
{
   Q_BUFFER far * b_head;
   Q_LINK *c_head, *t_head, *previous;
   Q_TRIGGER * tch;
   Q_TYPE type;
   int far * channel_buffer;
   unsigned int new_trigger_ctr, trigger_ctr;
   FLAG done;
   long offset;


   if (! trigger_enabled)
      return FALSE;

   b_head = dm_get_first_buffer();
   new_trigger_ctr = trigger_ctr = 0;

   while (b_head != NULL) {
      /* Clear out the ghosts */
      trigger_window_check (b_head->info.begintime);

      /* Scan buffer for any new triggers */
      c_head = channel_queue.head;
      while (c_head != NULL) {
         offset = b_head->info.blocksize * c_head->type.channel->channel;
         channel_buffer = b_head->data + b_head->info.blocksize * c_head->type.channel->channel;
         if (triggered_on_channel (c_head->type.channel, channel_buffer, b_head->info.blocksize)) {
            tch = (Q_TRIGGER *) calloc (1, sizeof(Q_TRIGGER));
            if (tch == NULL) er_abort (T_NO_STORAGE);

            tch->trigger_flag = FALSE;
            tch->trigger_time = b_head->info.begintime +
               (((double) trigger_location) / b_head->info.dig_rate);
            tch->qch = c_head->type.channel;
            type.trigger = tch;

            /* Put trigger in trigger list according to trigger time */
            t_head = trigger_queue.head;
            previous = NULL;
            done = FALSE;
            while (!done && t_head != NULL)
               if (tch->trigger_time < t_head->type.trigger->trigger_time) {
                  q_insert_link (&trigger_queue, previous, type);
                  done = TRUE;
               } else {
                  previous = t_head;
                  t_head = t_head->next;
               }
            if (!done) q_enqueue (&trigger_queue, type);
            new_trigger_ctr++;

         } else if (c_head->type.channel->info.trig_value == FALSE) {
            c_head->type.channel->info.lta = c_head->type.channel->info.abs_lta;
            c_head->type.channel->info.sta = c_head->type.channel->info.abs_sta;
         }
         c_head = c_head->next;
      }
      b_head = dm_get_next_buffer();
   }

   /* Are there any new triggers? */
   event_flag = FALSE;
   if (new_trigger_ctr) {
      t_head = trigger_queue.head;
      while (t_head != NULL) {
         if (t_head->type.trigger->qch->info.trig_value == OFF) {
            t_head->type.trigger->trigger_flag = TRUE;
            t_head->type.trigger->qch->info.trig_value = ON;
            t_head->type.trigger->qch->info.trig_time = t_head->type.trigger->trigger_time;
         }
         if (t_head->type.trigger->trigger_flag) {
            if (!event_flag && last_trigger < t_head->type.trigger->trigger_time) {
               last_trigger = t_head->type.trigger->trigger_time;
               lasttrig = t_head->type.trigger;
            }
            trigger_ctr++;
         }
         if (trigger_ctr >= t->critical_nu)
            event_flag = TRUE;       /* we have triggered an event */
         t_head = t_head->next;
      }
   }
   if (event_flag && lasttrig) lasttrig->qch->info.trig_value = EVENT_TRIGGER;
   s_display_triggers ();
   return (event_flag);
}
